import asyncio
import logging
import sys
from pathlib import Path

from aiogram import Bot, Dispatcher
from aiogram.enums import ParseMode
from aiogram.fsm.storage.memory import MemoryStorage
from aiogram.client.default import DefaultBotProperties
from aiohttp import web, ClientSession
from aiogram.webhook.aiohttp_server import SimpleRequestHandler, setup_application
from bot.handlers import admin_webhook

from config import BOT_TOKEN
from bot.utils.cleanup import cleanup_old_receipts
from bot.handlers import admin_notify
from bot.middlewares.membership import MembershipMiddleware
from bot.middlewares.contact_check import ContactCheckMiddleware
from bot.handlers import clearcache

from database.db import init_db, seed_rotation_groups

from bot.handlers.user import router as user_router
from bot.handlers.admin import router as admin_router
from bot.handlers.renew import router as renew_router
from bot.handlers import admin_discount

# ✅ برای وبهوک
from bot.utils.webhook_manager import load_config, get_full_url, save_config
from dotenv import load_dotenv
import os

# ✅ برای کرون تمدید اشتراک
from apscheduler.schedulers.asyncio import AsyncIOScheduler
from apscheduler.triggers.cron import CronTrigger
from pytz import timezone
from datetime import datetime
from bot.utils.cron import get_cron_status
from bot.utils.renewal_notifier import send_renewal_notifications

load_dotenv()
ADMINS = [int(x) for x in os.getenv("ADMINS", "").split(",") if x]
ADMIN_ID = ADMINS[0] if ADMINS else None

bot = Bot(
    token=BOT_TOKEN,
    default=DefaultBotProperties(parse_mode=ParseMode.HTML)
)
dp = Dispatcher(storage=MemoryStorage())


# ✅ تابع زمان‌بند تمدید اشتراک‌ها
async def start_renewal_cron(bot):
    scheduler = AsyncIOScheduler(timezone="Asia/Tehran")

    async def renewal_job():
        try:
            if await get_cron_status():
                print(f"⏰ [CRON] اجرای خودکار اطلاع‌رسانی تمدید در {datetime.now()}")
                result = await send_renewal_notifications(bot)
                print(f"✅ CRON Done → sent={result.get('sent_count', 0)}, failed={result.get('failed_count', 0)}")
            else:
                print("🚫 [CRON] غیرفعال است، اجرا نشد.")
        except Exception as e:
            print(f"⚠️ خطا در اجرای CRON: {e}")

    scheduler.add_job(
        renewal_job,
        CronTrigger(hour=3, minute=0, timezone=timezone("Asia/Tehran"))
    )
    scheduler.start()
    print("🟢 زمان‌بند تمدید اشتراک فعال شد (روزانه ساعت ۰۳:۰۰)")


# ✅ بررسی در دسترس بودن دامنه وبهوک
async def check_webhook_url(url: str) -> bool:
    try:
        async with ClientSession() as session:
            async with session.get(url, timeout=5) as resp:
                return resp.status in [200, 403, 405]
    except Exception:
        return False


async def main():
    # ✅ پیام در صورت ری‌استارت
    if "--restarted" in sys.argv:
        idx = sys.argv.index("--restarted")
        admin_id = int(sys.argv[idx + 1]) if idx + 1 < len(sys.argv) else None
        if admin_id:
            try:
                await bot.send_message(admin_id, "✅ ربات با موفقیت ری‌استارت شد و آماده به کار است.")
            except Exception as e:
                print(f"⚠️ ارسال پیام ری‌استارت به ادمین ناموفق بود: {e}")

    # ✅ تنظیم لاگ‌ها
    for handler in logging.root.handlers[:]:
        logging.root.removeHandler(handler)
    logging.basicConfig(level=logging.CRITICAL + 1)
    print("🚀 Starting bot...")

    # ✅ آماده‌سازی پایگاه داده
    await init_db()
    await seed_rotation_groups()
    Path("client_ph").mkdir(exist_ok=True)

    # ✅ پاک‌سازی هفتگی
    cleanup_old_receipts(folder="receipts", days=60)
    asyncio.create_task(schedule_cleanup())

    # ✅ میان‌افزارها
    dp.message.middleware(ContactCheckMiddleware())
    dp.message.middleware(MembershipMiddleware())
    dp.callback_query.middleware(MembershipMiddleware())

    # ✅ هندلرها
    dp.include_router(user_router)
    dp.include_router(admin_router)
    dp.include_router(renew_router)
    dp.include_router(admin_discount.router)
    dp.include_router(admin_notify.router)
    dp.include_router(clearcache.router)
    dp.include_router(admin_webhook.router)

    # ✅ کرون تمدید اشتراک
    await start_renewal_cron(bot)

    # ✅ بررسی و مدیریت خودکار وبهوک
    cfg = load_config()
    full_url = get_full_url()

    try:
        webhook_info = await bot.get_webhook_info()
        if webhook_info.url:
            print(f"🌐 وبهوک فعلی: {webhook_info.url}")
            if webhook_info.last_error_date:
                print(f"⚠️ آخرین خطا در وبهوک: {webhook_info.last_error_message}")
                print("🚫 حذف وبهوک و بازگشت به Polling ...")
                await bot.delete_webhook(drop_pending_updates=True)
                cfg["enabled"] = False
                save_config(cfg)
        else:
            print("ℹ️ هیچ وبهوکی تنظیم نشده است.")
    except Exception as e:
        print(f"⚠️ بررسی وبهوک با خطا مواجه شد: {e}")
        await bot.delete_webhook(drop_pending_updates=True)
        cfg["enabled"] = False
        save_config(cfg)

    # ✅ تلاش برای راه‌اندازی وبهوک در صورت فعال بودن
    if cfg.get("enabled") and full_url:
        print(f"🌐 تلاش برای اجرای Webhook در {full_url}")
        is_valid = await check_webhook_url(cfg["domain"])
        if not is_valid:
            print("⚠️ دامنه یا مسیر وبهوک در دسترس نیست. بازگشت به Polling...")
            cfg["enabled"] = False
            save_config(cfg)
            if ADMIN_ID:
                try:
                    await bot.send_message(ADMIN_ID, "⚠️ وبهوک در دسترس نبود. ربات به حالت Polling برگشت.")
                except:
                    pass
            await bot.delete_webhook(drop_pending_updates=True)
            await dp.start_polling(bot)
            return

        try:
            app = web.Application()
            SimpleRequestHandler(dispatcher=dp, bot=bot).register(app, path=cfg["path"])
            setup_application(app, dp, bot=bot)
            await bot.set_webhook(full_url)
            print("✅ وبهوک با موفقیت فعال شد.")
            web.run_app(app, host="0.0.0.0", port=cfg["port"])
        except Exception as e:
            print(f"❌ خطا در فعال‌سازی وبهوک: {e}")
            cfg["enabled"] = False
            save_config(cfg)
            if ADMIN_ID:
                try:
                    await bot.send_message(ADMIN_ID, f"❌ وبهوک دچار خطا شد:\n<code>{str(e)}</code>\n↩️ برگشت به Polling.", parse_mode="HTML")
                except:
                    pass
            await bot.delete_webhook(drop_pending_updates=True)
            await dp.start_polling(bot)
    else:
        print("🤖 Polling mode فعال شد...")
        await bot.delete_webhook(drop_pending_updates=True)
        await dp.start_polling(bot)


async def schedule_cleanup():
    while True:
        try:
            print("🧹 Running weekly cleanup task...")
            cleanup_old_receipts(folder="receipts", days=60)
        except Exception as e:
            print(f"⚠️ Cleanup error: {e}")
        await asyncio.sleep(7 * 24 * 60 * 60)


if __name__ == "__main__":
    asyncio.run(main())
