# database/models.py
from typing import Optional
from datetime import datetime
import enum

from sqlalchemy import (
    BigInteger,
    Float,
    ForeignKey,
    Integer,
    String,
    Enum,
    DateTime,
    Column,
    Boolean,
    text,
)
from sqlalchemy.ext.asyncio import AsyncAttrs
from sqlalchemy.orm import DeclarativeBase, Mapped, mapped_column, relationship

from sqlalchemy import Column, Integer, String, DateTime
from sqlalchemy.sql import func

# ---------- Base ----------
class Base(AsyncAttrs, DeclarativeBase):
    """Base class for all models."""
    pass




class UserPhone(Base):
    __tablename__ = "user_phones"

    id = Column(Integer, primary_key=True, autoincrement=True)
    user_id = Column(Integer, unique=True, nullable=False)
    phone_number = Column(String, nullable=False)
    username = Column(String, nullable=True)
    created_at = Column(DateTime(timezone=True), server_default=func.now())



# ---------- User ----------
class User(Base):
    __tablename__ = "users"

    user_id: Mapped[int] = mapped_column(BigInteger, primary_key=True)
    username:   Mapped[Optional[str]] = mapped_column(String(32))
    status:     Mapped[str] = mapped_column(String(20), default="active")
    invite_code: Mapped[Optional[str]] = mapped_column(String(20))
    referred_by: Mapped[Optional[int]] = mapped_column(BigInteger)

    gift_credit: Mapped[int] = mapped_column(
        Integer,
        nullable=False,
        default=0,
        server_default=text("0"),
    )

    orders: Mapped[list["Order"]] = relationship("Order", back_populates="user")

    user_count_requested: Mapped[int] = mapped_column(Integer, default=1, nullable=False)
    discount_applied: Mapped[int] = mapped_column(Integer, default=0, nullable=False)
    receipt: Mapped[Optional[str]] = mapped_column(String, nullable=True)

    def __repr__(self) -> str:
        return f"<User {self.user_id} ({self.username})>"


# ---------- Discount Code ----------
class DiscountCode(Base):
    __tablename__ = "discount_codes"

    id: Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    code: Mapped[str] = mapped_column(String(50), unique=True, nullable=False)   # کد تخفیف
    percent: Mapped[int] = mapped_column(Integer, nullable=False)               # درصد تخفیف
    expires_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True)  # تاریخ انقضا
    usage_limit: Mapped[Optional[int]] = mapped_column(Integer, nullable=True)  # حداکثر دفعات
    used_count: Mapped[int] = mapped_column(Integer, default=0)                 # دفعات استفاده شده
    per_user_limit: Mapped[Optional[int]] = mapped_column(Integer, nullable=True)

    def is_valid(self) -> bool:
        """Check if discount code is still valid."""
        if self.expires_at and self.expires_at < datetime.utcnow():
            return False
        if self.usage_limit and self.used_count >= self.usage_limit:
            return False
        return True

    def __repr__(self) -> str:
        return f"<DiscountCode {self.code} -{self.percent}%>"


class DiscountUsage(Base):
    __tablename__ = "discount_usages"

    id: Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    user_id: Mapped[int] = mapped_column(BigInteger, ForeignKey("users.user_id"))
    discount_id: Mapped[int] = mapped_column(Integer, ForeignKey("discount_codes.id"))
    order_id: Mapped[int] = mapped_column(Integer, ForeignKey("orders.id"))
    used_at: Mapped[datetime] = mapped_column(
        DateTime,
        default=datetime.utcnow,
        server_default=text("CURRENT_TIMESTAMP"),
    )


# ---------- PlanCategory ----------
class PlanCategory(Base):
    __tablename__ = "plan_categories"

    id:    Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    type:  Mapped[str] = mapped_column(String(20))          # حجمی / نامحدود
    title: Mapped[str] = mapped_column(String(50))
    description: Mapped[Optional[str]] = mapped_column(String(255))
    is_full: Mapped[bool] = mapped_column(Boolean, default=False, nullable=False)

    plans: Mapped[list["Plan"]] = relationship(
        back_populates="category", cascade="all, delete-orphan"
    )

    def __repr__(self) -> str:
        return f"<PlanCategory {self.id} {self.title}>"


# ---------- Plan ----------
class Plan(Base):
    __tablename__ = "plans"

    id:    Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    name:  Mapped[str] = mapped_column(String(50))
    type:  Mapped[str] = mapped_column(String(20))          # VIP / عادی …
    category_id: Mapped[int] = mapped_column(ForeignKey("plan_categories.id"))
    price: Mapped[float] = mapped_column(Float)
    description: Mapped[Optional[str]] = mapped_column(String(255))
    status:      Mapped[str] = mapped_column(String(20), default="active")
    user_limit:  Mapped[int] = mapped_column(Integer, default=1)
    volume_limit: Mapped[Optional[float]] = mapped_column(Float, nullable=True, default=None)

    orders:   Mapped[list["Order"]] = relationship("Order", back_populates="plan")
    category: Mapped["PlanCategory"] = relationship(back_populates="plans")
    is_full: Mapped[bool] = mapped_column(Boolean, default=False, nullable=False)

    def __repr__(self) -> str:
        return f"<Plan {self.id} {self.name} ({self.price})>"


# ---------- Enumها ----------
class PaymentStatus(enum.StrEnum):
    pending  = "pending"
    approved = "approved"
    rejected = "rejected"
    expired  = "expired"


# ---------- Order ----------
class Order(Base):
    __tablename__ = "orders"

    id:   Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    order_code: Mapped[str] = mapped_column(String(20), unique=True, nullable=False)
    user_id: Mapped[int] = mapped_column(BigInteger, ForeignKey("users.user_id"))
    plan_id: Mapped[int] = mapped_column(Integer, ForeignKey("plans.id"))
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.utcnow)

    start_date: Mapped[datetime] = mapped_column(
        DateTime, nullable=False,
        doc="تاریخ شروع اشتراک (محاسبه‌شده هنگام ایجاد سفارش)"
    )

    expiration_date: Mapped[datetime] = mapped_column(
        DateTime, nullable=False,
        doc="تاریخ پایان اشتراک (محاسبه‌شده هنگام ایجاد سفارش)"
    )

    assigned_link_url = Column(String, nullable=True)
    assigned_link_id = Column(Integer, ForeignKey("link_pool.id"), nullable=True)
    last_notified_at: Mapped[datetime | None] = mapped_column(DateTime, nullable=True)
    renewed_to = Column(Integer, ForeignKey("orders.id"), nullable=True)

    payment_method:  Mapped[str] = mapped_column(String(20), default="card_to_card")
    payment_status:  Mapped[PaymentStatus] = mapped_column(
        Enum(PaymentStatus), default=PaymentStatus.pending
    )
    receipt_file_id: Mapped[Optional[str]] = mapped_column(String(200))
    # در کلاس Order
    receipt_type: Mapped[Optional[str]] = mapped_column(String(10), nullable=True)  # "file" یا "text"
    receipt_value: Mapped[Optional[str]] = mapped_column(String, nullable=True)     # متن رسید یا file_id

    admin_note:      Mapped[Optional[str]] = mapped_column(String(255))
    config_sent:     Mapped[bool] = mapped_column(default=False)
    assigned_link:   Mapped[Optional[str]] = mapped_column(String(255), nullable=True)

    renew_order_id: Mapped[Optional[int]] = mapped_column(
        Integer, ForeignKey("orders.id"), nullable=True
    )

    plan: Mapped["Plan"] = relationship("Plan", back_populates="orders")
    user: Mapped["User"] = relationship("User", back_populates="orders")
    

    final_price:      Mapped[Optional[int]] = mapped_column(Integer, nullable=True)
    discount_percent: Mapped[Optional[int]] = mapped_column(Integer, nullable=True)
    discount_code: Mapped[Optional[str]] = mapped_column(String(50), nullable=True)

    def __repr__(self) -> str:
        return f"<Order {self.id} {self.user_id} status={self.payment_status}>"


# ---------- Unlimited Rotation Group ----------
class UnlimitedRotationGroup(Base):
    __tablename__ = "unlimited_rotation_groups"

    id:   Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    start_day: Mapped[int] = mapped_column(Integer)        # 0,5,10,15,20,25
    end_day:   Mapped[int] = mapped_column(Integer)        # start_day + 4
    discount_percent: Mapped[float] = mapped_column(Float, default=0.0)

    def __repr__(self) -> str:
        return f"<UnlimitedRotationGroup {self.start_day}-{self.end_day} (-{self.discount_percent}% )>"


# ---------- Link Pool ----------
class LinkType(enum.StrEnum):
    volume     = "volume"     # حجمی
    unlimited  = "unlimited"  # نامحدود


class LinkPool(Base):
    __tablename__ = "link_pool"

    id:       Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    url:      Mapped[str] = mapped_column(String(255))

    capacity:   Mapped[int] = mapped_column(Integer)
    used_users: Mapped[int] = mapped_column(Integer, default=0)
    status:     Mapped[str] = mapped_column(String(20), default="active")

    type: Mapped[LinkType] = mapped_column(
        Enum(LinkType), default=LinkType.volume,
        doc="حجمی یا نامحدود"
    )

    plan_id: Mapped[Optional[int]] = mapped_column(Integer, ForeignKey("plans.id"))

    admin_label: Mapped[Optional[str]] = mapped_column(String(100), nullable=True)
    admin_note:  Mapped[Optional[str]] = mapped_column(String(255), nullable=True)

    plan: Mapped["Plan"] = relationship("Plan")

    def __repr__(self) -> str:
        return (
            f"<LinkPool {self.id} {self.admin_label or self.url[:30]} "
            f"{self.type} cap={self.capacity} used={self.used_users}>"
        )



# database/models.py
class CronStatus(Base):
    __tablename__ = "cron_status"

    id: Mapped[int] = mapped_column(Integer, primary_key=True)
    is_active: Mapped[bool] = mapped_column(Boolean, default=False)  # وضعیت کرون (فعال یا غیرفعال)



# --- Settings (برای ذخیره وضعیت کرون) ---
class Setting(Base):
    __tablename__ = "settings"

    key: Mapped[str] = mapped_column(String(100), primary_key=True)
    value: Mapped[Optional[str]] = mapped_column(String(255), nullable=True)

    def __repr__(self) -> str:
        return f"<Setting {self.key}={self.value}>"


# --- Notification Log (برای جلوگیری از ارسال تکراری) ---
class NotificationLog(Base):
    __tablename__ = "notification_logs"

    id: Mapped[int] = mapped_column(Integer, primary_key=True, autoincrement=True)
    user_id: Mapped[int] = mapped_column(BigInteger, nullable=False)
    order_id: Mapped[int] = mapped_column(Integer, nullable=False)
    notification_type: Mapped[str] = mapped_column(String(50), nullable=False)
    sent_at: Mapped[datetime] = mapped_column(
        DateTime,
        default=datetime.utcnow,
        server_default=text("CURRENT_TIMESTAMP"),
    )

    def __repr__(self) -> str:
        return f"<NotificationLog {self.id} user={self.user_id} order={self.order_id} type={self.notification_type}>"


# تشخیص و ذخیره کاربرانی که ربات را بلاک کرده اند 
class BlockedUser(Base):
    __tablename__ = "blocked_users"

    user_id: Mapped[int] = mapped_column(BigInteger, primary_key=True)
    blocked_at: Mapped[datetime] = mapped_column(
        DateTime,
        default=datetime.utcnow
    )


# مازولار کردن اطلاعات پرداخت در پنل ادمین 
class PaymentConfig(Base):
    __tablename__ = "payment_config"

    id: Mapped[int] = mapped_column(primary_key=True, default=1)
    card_number: Mapped[str] = mapped_column(String(32))
    card_holder: Mapped[str] = mapped_column(String(128))
    updated_at: Mapped[datetime] = mapped_column(
        DateTime,
        default=datetime.utcnow,
        onupdate=datetime.utcnow
    )
