"""
ماژول ارسال اعلان تمدید برای کاربران
"""
import asyncio
import time
from datetime import datetime, timedelta
from aiogram import Bot
from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton
from database.models import Order, User
from database.db import async_session

import jdatetime


# ✅ مقادیر اولیه throttling ساده
THROTTLE_SECONDS = 2.0      # فاصله بین هر ارسال پیام
CHUNK_SIZE = 30             # ارسال پیام‌ها در دسته‌های ۵۰ تایی
CHUNK_PAUSE = 9.0           # توقف بین دسته‌ها (۵ ثانیه)

async def send_renewal_notifications(bot: Bot):
    """
    ارسال اعلان تمدید برای کاربرانی که اشتراک‌شان در چند روز آینده منقضی می‌شود.
    """

    from sqlalchemy import select

    sent_count = 0
    failed_count = 0

    start_time = time.time()   # زمان شروع عملیات

    async with async_session() as session:
        now = datetime.utcnow()
        target_time = now + timedelta(days=5)

        result = await session.execute(
            select(Order).where(
                Order.expiration_date <= target_time,
                Order.expiration_date > now
            )
        )
        orders = result.scalars().all()
        total_orders = len(orders)

        print(f"🧩 بررسی {total_orders} سفارش برای تمدید...")

        counter = 0

        for order in orders:
            try:
                user = await session.get(User, order.user_id)
                if not user:
                    continue

                order_identifier = getattr(order, "order_code", None) or getattr(order, "custom_id", None) or str(order.id)

                remaining_days = (order.expiration_date - now).days
                persian_date = jdatetime.datetime.fromgregorian(datetime=order.expiration_date).strftime("%Y/%m/%d")

                message = (
                    f"⏰ اشتراک شما با شناسه سفارش <b>{order_identifier}</b>\n"
                    f"تا تاریخ <b>{persian_date}</b> معتبر است.\n"
                    f"📅 تعداد روز باقی‌مانده: <b>{remaining_days} روز</b>\n\n"
                    "✅ برای جلوگیری از قطع سرویس، لطفاً برای تمدید اشتراک اقدام کنید ."
                )

                keyboard = InlineKeyboardMarkup(
                    inline_keyboard=[
                        [InlineKeyboardButton(text="🔄 تمدید سرویس", callback_data=f"renew:{order.id}")]
                    ]
                )

                await bot.send_message(order.user_id, message, reply_markup=keyboard)

                order.last_notified_at = datetime.utcnow()
                await session.commit()

                sent_count += 1
                counter += 1

                # ⏳ فاصله‌ی ارسال‌ها
                await asyncio.sleep(THROTTLE_SECONDS)

                # 🧠 توقف کوتاه بین دسته‌ها
                if counter % CHUNK_SIZE == 0:
                    print(f"⏸ توقف {CHUNK_PAUSE} ثانیه برای کاهش فشار روی سرور")
                    await asyncio.sleep(CHUNK_PAUSE)

            except Exception as e:
                print(f"⚠️ خطا برای کاربر {order.user_id}: {e}")
                failed_count += 1

    duration = round(time.time() - start_time, 2)

    print(f"✅ پایان عملیات: موفق: {sent_count} خطا: {failed_count} زمان: {duration}s")

    return {
        "sent_count": sent_count,
        "failed_count": failed_count,
        "total_checked": total_orders,
        "duration": duration
    }


if __name__ == "__main__":
    print("این فایل نباید اجرا شود.")
