import tarfile
import gzip
import shutil
import hashlib
import sqlite3
import json
import asyncio
from pathlib import Path
from datetime import datetime
from database.db import engine


PROJECT_DIR = Path(__file__).resolve().parent.parent.parent
DATA_DIR = PROJECT_DIR / "data"
BACKUP_DIR = DATA_DIR / "backups"
DB_PATH = DATA_DIR / "database.db"

BACKUP_DIR.mkdir(parents=True, exist_ok=True)


def calculate_sha256(file_path: Path) -> str:
    sha256 = hashlib.sha256()
    with open(file_path, "rb") as f:
        for chunk in iter(lambda: f.read(8192), b""):
            sha256.update(chunk)
    return sha256.hexdigest()


def get_db_revision(db_path: Path) -> str | None:
    try:
        with sqlite3.connect(db_path) as conn:
            cursor = conn.cursor()
            cursor.execute("SELECT version_num FROM alembic_version LIMIT 1")
            row = cursor.fetchone()
            return row[0] if row else None
    except Exception:
        return None


async def create_full_backup(progress_callback=None) -> Path:
    """
    ساخت بکاپ کامل با metadata و hash
    hash داخل metadata = SHA256 فقط محتوای اصلی (بدون backup_meta.json)
    """
    await engine.dispose()  # بستن connectionهای sqlalchemy

    timestamp = datetime.utcnow().strftime("%Y%m%d_%H%M%S")
    temp_dir = BACKUP_DIR / f"temp_backup_{timestamp}"
    temp_dir.mkdir(exist_ok=True)

    try:
        # 1. آماده‌سازی دیتابیس
        if progress_callback:
            await progress_callback("در حال گرفتن snapshot دیتابیس...")
        db_copy = temp_dir / f"database_{timestamp}.db"
        with sqlite3.connect(DB_PATH) as src:
            with sqlite3.connect(db_copy) as dst:
                src.backup(dst)

        # 2. فشرده‌سازی دیتابیس
        if progress_callback:
            await progress_callback("فشرده‌سازی دیتابیس...")
        compressed_db = temp_dir / f"database_{timestamp}.db.gz"
        with open(db_copy, "rb") as fin, gzip.open(compressed_db, "wb") as fout:
            shutil.copyfileobj(fin, fout)
        db_copy.unlink()

        # 3. کپی پوشه‌های اضافی
        folders_to_backup = [
            ("receipts", PROJECT_DIR / "receipts"),
            ("data/reports", PROJECT_DIR / "data/reports"),
        ]
        included_folders = []
        for arcname, src_path in folders_to_backup:
            if src_path.exists():
                dest = temp_dir / arcname
                shutil.copytree(src_path, dest, dirs_exist_ok=True)
                included_folders.append(arcname)

        # 4. ساخت metadata اولیه (hash هنوز محاسبه نشده)
        meta = {
            "created_at_utc": datetime.utcnow().isoformat(),
            "timestamp": timestamp,
            "schema_revision": get_db_revision(DB_PATH),
            "sqlite_version": sqlite3.sqlite_version,
            "included_folders": included_folders,
            "sha256": None,
        }
        meta_path = temp_dir / "backup_meta.json"
        meta_path.write_text(json.dumps(meta, indent=2, ensure_ascii=False), encoding="utf-8")

        # ───────────────────────────────────────────────
        # مرحله کلیدی: محاسبه hash فقط محتوای اصلی (بدون metadata.json)
        # ───────────────────────────────────────────────
        archive_path = BACKUP_DIR / f"full_backup_{timestamp}.tar.gz"

        if progress_callback:
            await progress_callback("محاسبه امضای محتوای اصلی...")

        temp_content_archive = BACKUP_DIR / f"temp_content_{timestamp}.tar.gz"

        with tarfile.open(temp_content_archive, "w:gz") as tar:
            for item in sorted(temp_dir.iterdir()):
                if item.name == "backup_meta.json":
                    continue
                tar.add(item, arcname=item.name)

        content_hash = calculate_sha256(temp_content_archive)

        # پاک کردن فایل موقت محتوا
        temp_content_archive.unlink(missing_ok=True)

        # 5. آپدیت metadata با hash محتوا
        meta["sha256"] = content_hash
        meta_path.write_text(json.dumps(meta, indent=2, ensure_ascii=False), encoding="utf-8")

        # 6. ساخت آرشیو نهایی کامل (شامل metadata با hash درست)
        if progress_callback:
            await progress_callback("ساخت آرشیو نهایی...")

        with tarfile.open(archive_path, "w:gz") as tar:
            for item in sorted(temp_dir.iterdir()):
                tar.add(item, arcname=item.name)

        final_hash = calculate_sha256(archive_path)
        

        if progress_callback:
            await progress_callback("بکاپ با موفقیت ساخته شد.")

        return archive_path

    except Exception as e:
        raise

    finally:
        shutil.rmtree(temp_dir, ignore_errors=True)