import os
import sys
import asyncio
from aiogram import Router, F
from aiogram.types import CallbackQuery, Message
from aiogram.fsm.state import State, StatesGroup
from aiogram.fsm.context import FSMContext
from aiogram.utils.keyboard import InlineKeyboardBuilder
from bot.utils.webhook_manager import (
    load_config, save_config, activate_webhook, deactivate_webhook, get_full_url
)

router = Router()

class WebhookSetup(StatesGroup):
    waiting_for_domain = State()
    waiting_for_path = State()
    waiting_for_port = State()

# ✅ تابع ری‌استارت خودکار ربات
async def restart_bot(callback: CallbackQuery):
    await callback.message.answer("♻️ ربات در حال ری‌استارت برای اعمال تغییرات وبهوک است...")
    await asyncio.sleep(1.5)
    try:
        python_exec = sys.executable
        # ری‌استارت با ارسال آرگومان ادمین برای اطلاع‌رسانی
        os.execl(python_exec, python_exec, *sys.argv, "--restarted", str(callback.from_user.id))
    except Exception as e:
        await callback.message.answer(f"⚠️ خطا در ری‌استارت ربات: {e}")

@router.callback_query(F.data == "admin:webhook_menu")
async def webhook_menu(callback: CallbackQuery):
    cfg = load_config()
    kb = InlineKeyboardBuilder()
    kb.button(text="🌐 تنظیم دامنه", callback_data="admin:webhook:set_domain")
    kb.button(text="🛠 تنظیم مسیر", callback_data="admin:webhook:set_path")
    kb.button(text="⚙️ تنظیم پورت", callback_data="admin:webhook:set_port")
    kb.button(text="✅ فعال‌سازی", callback_data="admin:webhook:enable")
    kb.button(text="❌ غیرفعال‌سازی", callback_data="admin:webhook:disable")
    kb.button(text="🔙 بازگشت", callback_data="admin:renew_notify_menu")
    kb.adjust(2)

    status = "🟢 فعال" if cfg["enabled"] else "🔴 غیرفعال"
    await callback.message.edit_text(
        f"📡 <b>مدیریت وبهوک</b>\n"
        f"وضعیت فعلی: {status}\n\n"
        f"🌐 دامنه: <code>{cfg['domain'] or '-'}</code>\n"
        f"🛠 مسیر: <code>{cfg['path']}</code>\n"
        f"⚙️ پورت: <code>{cfg['port']}</code>",
        parse_mode="HTML",
        reply_markup=kb.as_markup()
    )

# ✅ تنظیم دامنه
@router.callback_query(F.data == "admin:webhook:set_domain")
async def ask_domain(callback: CallbackQuery, state: FSMContext):
    await state.set_state(WebhookSetup.waiting_for_domain)
    await callback.message.answer("🌐 لطفاً دامنه کامل وبهوک را وارد کنید (مثل https://example.com):")

@router.message(WebhookSetup.waiting_for_domain)
async def save_domain(message: Message, state: FSMContext):
    cfg = load_config()
    cfg["domain"] = message.text.strip()
    save_config(cfg)
    await state.clear()
    await message.answer(f"✅ دامنه ثبت شد:\n<code>{cfg['domain']}</code>", parse_mode="HTML")

# ✅ تنظیم مسیر
@router.callback_query(F.data == "admin:webhook:set_path")
async def ask_path(callback: CallbackQuery, state: FSMContext):
    await state.set_state(WebhookSetup.waiting_for_path)
    await callback.message.answer("🛠 لطفاً مسیر (path) وبهوک را وارد کنید (مثل /botwebhook):")

@router.message(WebhookSetup.waiting_for_path)
async def save_path(message: Message, state: FSMContext):
    cfg = load_config()
    cfg["path"] = message.text.strip()
    save_config(cfg)
    await state.clear()
    await message.answer(f"✅ مسیر ثبت شد:\n<code>{cfg['path']}</code>", parse_mode="HTML")

# ✅ تنظیم پورت
@router.callback_query(F.data == "admin:webhook:set_port")
async def ask_port(callback: CallbackQuery, state: FSMContext):
    await state.set_state(WebhookSetup.waiting_for_port)
    await callback.message.answer("⚙️ لطفاً پورت را وارد کنید (مثلاً 8080):")

@router.message(WebhookSetup.waiting_for_port)
async def save_port(message: Message, state: FSMContext):
    cfg = load_config()
    try:
        cfg["port"] = int(message.text.strip())
        save_config(cfg)
        await state.clear()
        await message.answer(f"✅ پورت ثبت شد:\n<code>{cfg['port']}</code>", parse_mode="HTML")
    except ValueError:
        await message.answer("❌ لطفاً یک عدد معتبر وارد کنید.")

# ✅ فعال‌سازی وبهوک
@router.callback_query(F.data == "admin:webhook:enable")
async def enable_webhook(callback: CallbackQuery):
    success, msg = await activate_webhook(callback.bot)
    await callback.answer(msg, show_alert=True)
    await callback.message.answer("✅ وبهوک فعال شد. در حال ری‌استارت...")
    await restart_bot(callback)

# ✅ غیرفعال‌سازی وبهوک
@router.callback_query(F.data == "admin:webhook:disable")
async def disable_webhook(callback: CallbackQuery):
    success, msg = await deactivate_webhook(callback.bot)
    await callback.answer(msg, show_alert=True)
    await callback.message.answer("❌ وبهوک غیرفعال شد. در حال ری‌استارت...")
    await restart_bot(callback)
